#include "NetFlow9Processor.h"

unsigned int NetFlow9Processor::ByteTouInt32(unsigned char bytes[], int base) {
	//TODO: Check for out of range muppets
	//TODO: Endianess issues?
	return (bytes[base] << 24) + (bytes[base+1] << 16) + (bytes[base+2] << 8) + bytes[base+3];
}

unsigned short NetFlow9Processor::ByteTouInt16(unsigned char bytes[], int base) {
	//TODO: Check for out of range muppets
	//TODO: Endianess issues?
	return (bytes[base] << 8) + bytes[base+1];
}


NetFlow9Processor::NetFlow9Processor(struct sockaddr *remoteAddress, socklen_t *addressLength) {
	dataTemplateCache = new NetFlowDataTemplateCache();
	switch (remoteAddress->sa_family) {
		case AF_INET:
			//set IP
			inet_ntop(AF_INET, &((sockaddr_in *)remoteAddress)->sin_addr, remoteIPStr, INET_ADDRSTRLEN);
			break;
		case AF_INET6:
			//set IP
			break;
		default:
			//uhoh
			break;
	}
	name = "undefined";
	
}

FlowSet* NetFlow9Processor::ProcessPacket(unsigned char buffer[], int bufferLength) {

	int nextIndex = 0;					// Next byte in the buffer
	int version = 0;					// NetFlow version
	int count = 0;						// Records in the packet
	unsigned short flowSetID = 0;		// ID of the current FlowSet
	unsigned short flowSetLength = 0;	// Bytes in the current FlowSet
	unsigned int uptime = 0;			// Router uptime in miliseconds
	unsigned int packetTime = 0;		// Time sent by router (Unix seconds)
	unsigned int sequenceID = 0;		// Incremented by one for each packet sent
	unsigned int sourceID = 0;			// Locally significant source ID
	FlowSet* processedFlowSet = nullptr;

	if ( bufferLength < 24) {
		//Packet is too short to contain a header
		return processedFlowSet;
	}
	
	//First 2 bytes (0 and 1) are the version
	version = (buffer[0] << 8 ) + buffer [1];
	if (version != 9) {
			//Not a NetFlow v9 packet
			return processedFlowSet;
	}

	//Bytes 2 and 3 are the count of records in the packet
	count = (buffer[2] << 8 ) + buffer [3];

	if (count < 1) {
			//Not a NetFlow v9 packet
			return processedFlowSet;
	}

	nextIndex = 4;
	uptime = ByteTouInt32(buffer,nextIndex);
	nextIndex += 4;
	packetTime = ByteTouInt32(buffer,nextIndex);
	nextIndex += 4;
	sequenceID = ByteTouInt32(buffer,nextIndex);
	nextIndex += 4;
	sourceID = ByteTouInt32(buffer,nextIndex);
	nextIndex += 4;

	do {
		flowSetID = ByteTouInt16(buffer,nextIndex);
		flowSetLength = ByteTouInt16(buffer,nextIndex+2);
		if (nextIndex + flowSetLength > bufferLength) {
			//Invalid/unsupported packet
			return processedFlowSet;
		}
		if (flowSetID > 255) {
			//DATA FlowSet
			printf("  -> Found data flowSetID %d, length %d.\n", flowSetID, flowSetLength);
			processedFlowSet = processDataFlowSet(buffer,nextIndex,flowSetID,flowSetLength, remoteIPStr);
		}
		else if (flowSetID == 0) {
			//TEMPLATE FlowSet
			printf("  -> Found template flowSetID %d, length %d.\n", flowSetID, flowSetLength);
			processTemplateFlowSet(buffer,nextIndex,flowSetLength);
		}
		else if (flowSetID == 1) {
			//OPTIONS FlowSet
			printf("  -> Found options flowSetID %d, length %d.\n", flowSetID, flowSetLength);
		}
		else {
			//Invalid/unsupported packet
			return processedFlowSet;
		}
		nextIndex += flowSetLength;
	}
	while ( nextIndex < bufferLength);

	return processedFlowSet;
}

FlowSet* NetFlow9Processor::processDataFlowSet(unsigned char buffer[], int startIndex, unsigned short flowSetID, unsigned short flowSetLength, char strRemoteIP[]) {
	int nextIndex = startIndex + 4;             // Next place to look (skip 4 header bytes already parsed)
	int endIndex = startIndex + flowSetLength ; // End the loop at the end of the template
	NetFlowDataTemplate *dataTemplate;          // Data Template object
	NetFlowDataTemplate::Field_t field;
	FlowSet *Flowset;
	int recordsExpected = 0;
	int dataTemplateRecordLength = 0;
	int lengthOfAllRecords = 0;
	int count = 0;

		dataTemplate = dataTemplateCache->GetDataTemplate(flowSetID);
		if (!dataTemplate) {
			printf("  -> Template ID %d doesn't exist in the cache...  skipping\n", flowSetID);
			return nullptr;
		}

		dataTemplateRecordLength = dataTemplate->recordLength();
		recordsExpected = flowSetLength / dataTemplateRecordLength;
		lengthOfAllRecords = recordsExpected * dataTemplateRecordLength;

		if (lengthOfAllRecords>flowSetLength) {
			printf("  -> Length of all record expected in the FlowSet exceeds the length reported\n");
			return nullptr;
		}

		printf("  -> FS Length %d, FSR Length: %d, FSRs expected: %d, recalculated length: %d\n", dataTemplateRecordLength, flowSetLength, recordsExpected, lengthOfAllRecords);
		
		Flowset = new FlowSet(strRemoteIP)

		for (count = 0; count < recordsExpected; count++) {
		
			printf("  -> Parsing record %d.\n", count);

			FlowSetRecord *record = new FlowSetRecord(dataTemplate->Fields.size());

			for (std::list<NetFlowDataTemplate::Field_t>::iterator it=dataTemplate->Fields.begin(); it != dataTemplate->Fields.end(); ++it) {
				field = *it;
				printf("    -> Code: %d, length: %d\n", field.code, field.length);

				//Form a standard record and return it

				unsigned char recordBuffer[field.length];
				std::copy(buffer + nextIndex, buffer + nextIndex + field.length, recordBuffer);
				record->AddRecord(field.code, field.length, recordBuffer);

				nextIndex += field.length;
			}

			Flowset->AddFlowSetRecord(record);
		}

	return Flowset;
}

void NetFlow9Processor::processTemplateFlowSet(unsigned char buffer[], int startIndex, unsigned short flowSetLength) {

	int nextIndex = startIndex + 4;				// Next place to look (skip 4 header bytes already parsed)
	int endIndex = startIndex + flowSetLength ; // End the loop at the end of the template
	unsigned short count = 0;					// Count of fields to extract
	unsigned short id = 0;						// ID the template applies to
	NetFlowDataTemplate *dataTemplate;			// Data Template object
	NetFlowDataTemplate *existingDataTemplate;	// Data Template object if it already exists
	NetFlowDataTemplate::Field_t field;

	do {

		//Lets get the Template ID and the count of fields
		id = ByteTouInt16(buffer,nextIndex);
		count = ByteTouInt16(buffer,nextIndex+2);

		printf("  -> Template ID %d\n", id);
	
		//Increment nextIndex to skip the 4 bytes of the header already parsed
		nextIndex += 4;

		dataTemplate = new NetFlowDataTemplate(id, count);

		int i;

		for (i=0; i < count; i++) {
			field.code = ByteTouInt16(buffer,nextIndex);
			field.length = ByteTouInt16(buffer,nextIndex+2);
			printf("    -> Field %d. Field Code %d, length %d\n", i, field.code, field.length );
			dataTemplate->Fields.push_back( field );
			nextIndex += 4;
		}

		//Check if ID is in cache
		if ( dataTemplateCache->ExistsDataTemplate(id) ) {
			//Yes --> Update
			printf("  -> Template ID %d in cache.\n", id);
			existingDataTemplate = dataTemplateCache->DeleteDataTemplate(id);
			delete existingDataTemplate;
			if ( dataTemplateCache->AddDataTemplate(id, dataTemplate) ) {
				printf("  -> Template ID %d updated.\n", id);
			}
			else {
				printf("  -> ERROR: Failed to update Template ID %d.\n", id);
			}
		}
		else {
			//No --> Add
			if ( dataTemplateCache->AddDataTemplate(id, dataTemplate) ) {
				printf("  -> Template ID %d added.\n", id);
			}
			else {
				printf("  -> ERROR: Failed to add Template ID %d.\n", id);
			}
		}


	}
	while ( nextIndex < endIndex );

}
